/*
# mrtg-nrpe v2.8
#
# Compile with:
#	cc -s -O -o mrtg-nrpe mrtg-nrpe.c utils.c -lssl
# Tested with GNU gcc under RedHat Enterprise Linux
#
# S Shipway - www.steveshipway.org
# This is released under the GNU GPL.  See nsclient.ready2run.nl
# to obtain the NetSaint client for your Windows server!
#
# Perl script to collect information from remote NRPE
# client, and output in format suitable for MRTG.
#
# Usage:
#   mrtg-nrpe -H host [ -p port ] [ -n ] [ -t timeout ] [ -x ]
#       -c <module> [ -a <option> ... ] [ -o <offset> ]
#       [ -c <module> ] [ -a <option> ... ] [ -o <offset> ] 
#       [ -C ] (not yet enabled)
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <getopt.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <errno.h>
#include <netinet/in.h>
#include <netdb.h>
#include <fcntl.h>
#include <signal.h>
#include "common.h"
#include "utils.h"

#undef DEBUG

#define VERSION "2.8(C)"
#define PORT    5666
#define TIMEOUT 10
#define MAXPARSE 10 /* max number of numbers parsed from response */
#define MAXARGS  16 /* max number of function args to pass: NRPE says 16 */
#define CACHEFILE "/var/tmp/nrpe-c.cache"
#define RESPSIZE 128 /* size of an NRPE response */
#define MAXAGE   240 /* how many seconds old a cache entry can be */
#define MAXARGLEN 64
#define MAXCACHE  2048 /* cache may not grow bigger than this many items */

#define WITHALARM
#undef  WITHNONBLOCK
#define WITHSSL

/* #################################################################### */

#ifdef WITHSSL
#include <openssl/rsa.h>
#include <openssl/crypto.h>
#include <openssl/dh.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#endif

/* for the cache file */
struct cache {
	char host[64];
	char cmd[32];
	time_t time;
	char resp[RESPSIZE];
};
struct cachechain {
	struct cache entry;
	struct cachechain *next;
};
struct cachechain *cache = 0;
char  cachefile[128];

char   mesg[RESPSIZE];
double resp[2] = {0,0};
int    unknown[2] = {1,1};
int    sock = -1;
char   host[64];
char   pass[16];
int    port;
char * args[2][MAXARGS] = {
	{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
	{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}} ;
char * cmd[2] = {0,0}; /* commands */
int    offset[2] = {0,1}; /* unless a new command is specified */
int    timeout = TIMEOUT;
int    perfmode = 0;
#ifdef DEBUG
int    debugmode = 1;
#else
int    debugmode = 0;
#endif
int    cachemode = 0;
int    newdata   = 0;

#ifdef WITHSSL
SSL_METHOD *meth;
SSL_CTX *ctx;
SSL *ssl;
int usessl=1;
#else
int usessl=0;
#endif

/* cache functions.  This is SO much easier in perl...  */
void clearentry( struct cachechain *c ) {
	if(!c) { return; }
	if(c->next) { clearentry(c->next); }
	free(c);
}
void clearcache() {
	if(!cachemode)    { return; }
	if(debugmode) { printf("Clearing cache...\n"); }
	clearentry( cache );
	cache = (struct cachechain *)0;
}
void readcache() {
	FILE *fp;
	struct timeval tv;
	struct cachechain *c,*p;
	int i = 0;
	if(debugmode) { printf("called readcache()\n"); }
	if(!cachemode)    { return; }
	if(!cachefile[0]) { return; }
	clearcache();
	gettimeofday(&tv,NULL);

	/* open file for reading; read lock; input list, checking times */
	if(debugmode) { printf("Reading cache [%s]\n",cachefile); }
	fp = fopen( cachefile, "rb" );
	if(!fp) { 
		if(debugmode) { perror("Unable to open file: "); }
		return; }
	if( flock( fileno(fp), LOCK_SH ) ) { 
		if(debugmode) { perror("Unable to lock file: "); }
		fclose(fp); return; }
	c = (struct cachechain *)malloc(sizeof(struct cachechain));
	p = (struct cachechain *)0;
	while( fread( &(c->entry), sizeof(struct cache), 1, fp ) == 1 ) {
		c->next =  (struct cachechain *)0;
		if((tv.tv_sec - c->entry.time)>MAXAGE) { 
			if(debugmode) { 
				printf("Item too old: %lu > %d\n",
					(tv.tv_sec-c->entry.time), MAXAGE);
			}
			continue; 
		}
		if(debugmode) { printf("Adding %s/%s age %lus to cache\n",c->entry.host,
			c->entry.cmd,(tv.tv_sec-c->entry.time)); }
		if(cache) { p->next = c; p = c; } else { p = cache = c; }
		i++;
		c = (struct cachechain *)malloc(sizeof(struct cachechain));
	}
	fclose(fp);
	if(debugmode) { printf("%d cached items loaded.\n",i); }
}
void writecache() {
	FILE *fp;
	struct cachechain *c;
	int i = 0;

	if(!cachemode)    { return; }
	if(!cachefile[0]) { return; }
	if(!cache)        { return; }

	/* open file for writing and truncated; lock; output whole list */
	if(debugmode) { printf("Writing cache [%s]\n",cachefile); }
	fp = fopen( cachefile, "wb" );
	if(!fp) { return; }
	if( flock( fileno(fp), LOCK_EX ) ) { fclose(fp); return; }
	for( c = cache; c; c = c->next ) {
		fwrite( &(c->entry) , sizeof(struct cache), 1, fp);
		i++;
		if(i>MAXCACHE) { break; } /* catch runaways */
	}
	fclose(fp);
	if(debugmode) { printf("%d items saved.\n",i); }
}
char * qcache(char *command) {
	struct cachechain * c;
	char *response;

	if(!cachemode)    { return((char *)0); }
	if(!cache)        { return((char *)0); }
	if(debugmode) { printf("Searching cache for %s/%s\n",host,command); }
	for( c = cache; c ; c = c->next ) {
		if(!strcmp(c->entry.host,host) && !strcmp(c->entry.cmd,command)){
			break;
		}
	}
	if(c) { /* matched */
		if(debugmode) { printf("Cache match found!\n"); }
		response = (char *)malloc(RESPSIZE);
		strncpy(response,c->entry.resp,RESPSIZE);
		return(response);
	}
	return ((char*)0);
}
int cacheadd(char *command, char *response) {
	struct timeval tv;
	struct cachechain * c;
	if(!cachemode)    { return; }
	readcache();
	gettimeofday(&tv,NULL);
	c = (struct cachechain *)malloc(sizeof(struct cachechain));
	c->next = cache; cache = c; /* hook in at head */
	c->entry.time = tv.tv_sec;
	strncpy(c->entry.host,host,sizeof(c->entry.host));
	strncpy(c->entry.cmd,command,sizeof(c->entry.cmd));
	strncpy(c->entry.resp,response,RESPSIZE);
	writecache();
}

/*
######################################################################
*/
void outputresp() {
	if( unknown[0] ) { printf("UNKNOWN\n"); }
	else { printf("%f\n",resp[0]); }
	if( unknown[1] ) { printf("UNKNOWN\n"); }
	else { printf("%f\n",resp[1]); }
	printf("\n%s\n",mesg);
}
void dohelp() {
	printf("mrtg-nrpe -H host [ -p port ] [ -n ] [ -t timeout ] [ -x ]\n");
	printf("    -c <module> [ -a <arg> ... ] [ -o <offset> ]\n");
	printf("  [ -c <module> [ -a <arg> ... ]] [ -o <offset> ]\n");
	printf("-n : No SSL\n-x : Use perfparse section instead of normal text, if available\n");
	printf("-t : Seconds timeout (default %d)\n",TIMEOUT);
	printf("-p : NRPE port (default %d)\n",PORT);
	printf("-o : Specify which number in output to use (first is 0)\n");
	printf("You can specify zero or more arguments using -a.  If you do not specify a\nsecond command, then it defaults to the first.  If you only specify a second\noffset, then it uses the cached output of the first command.  If you specify\nneither a second command nor a second offset, then the second offset defaults\nto 1 while the first offset defaults to 0.\n");
	printf("The returned output is parsed and all numbers are extracted into a list.\nThe offset defines which will be returned.  If the status is 'UNKNOWN' then\nthe output is ignored.\n");
	exit(0);
}
/*
############################################
# Open the connection
*/
void makesocket() {
	struct hostent * hp;
	struct sockaddr_in ss;
	struct in_addr ia;
	int n,v;

	if(debugmode) { printf("Opening connection to host...\n");fflush(NULL);}

	hp = gethostbyname(host);
	if(!hp) {
		sprintf(mesg,"Unable to resolve %s",host);
		outputresp(); exit(1);
	}
	bzero(&ss,sizeof(ss));
	memcpy(&ss.sin_addr,hp->h_addr,sizeof(ss.sin_addr));
	ss.sin_family = AF_INET;
	ss.sin_port = htons(port);
	if(debugmode) { printf("Creating socket...\n");fflush(NULL);}
	sock = socket(PF_INET,SOCK_STREAM,0);
	if(sock<0) {
		sprintf(mesg,"Unable to create socket");
		outputresp(); exit(1);
	}
#ifdef WITHNONBLOCK
#if defined(O_NONBLOCK)
    if (-1 == (v = fcntl(sock, F_GETFL, 0)))
        v = 0;
    fcntl(sock, F_SETFL, v | O_NONBLOCK);
#else
    /* Otherwise, use the old way of doing it */
    v = 1;
    ioctl(sock, FIOBIO, &v);
#endif
#endif

	/* this can hang */
	if(debugmode) { printf("Connecting...\n");fflush(NULL);}
	n = connect(sock,(struct sockaddr *)&ss,(socklen_t)sizeof(ss));
	if(n) {
		sprintf(mesg,"Unable to connect (%d)",errno);
		outputresp(); exit(1);
	}
	if(debugmode) { printf("Setting reuseaddr...\n");fflush(NULL);}
	v = 1;
	setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,&v,sizeof(v));
}
/*
######################################################################
# command no, optional argument, where to store value
# returns 0 on success
# command, list of args, where to put value, offset, use cached?
*/
int ask(char *cmd,char **arg,double *rvp,int o,int perfmode,int cached) {
	static double a[MAXPARSE]; 
	static int parsed = 0;
	char * buf;
	int i;
	char *s1, *s2;
    u_int32_t packet_crc32;
    u_int32_t calculated_crc32;
    packet send_packet;
    packet receive_packet;
    int bytes_to_send;
    int bytes_to_recv;
	int rc;
	int result;

	if(cached) {
		if(o>parsed) { return(1); }
		*rvp = a[o];
	} else {
		if( debugmode ) { printf( "Seeking offset %d\n",o); }
		if( cachemode ) {
			buf = qcache( cmd );
		} else { buf = (char *)0; parsed = 0;  }
		if(buf) {
			if(debugmode) { 
				printf("Skipping actual query because cache worked\n"); }
		} else {

		/* make connection */
		makesocket();
		if(sock<0) { printf("Error: socket wrong?\n"); exit(1); }

#ifdef WITHSSL
 	   /* do SSL handshake */
	    if(usessl){
			if(debugmode) { printf("SSL handshake\n"); fflush(NULL); }
	        if((ssl=SSL_new(ctx))!=NULL){
	            SSL_CTX_set_cipher_list(ctx,"ADH");
	            SSL_set_fd(ssl,sock);
	            if((rc=SSL_connect(ssl))!=1){
	            	SSL_CTX_free(ctx);
	                sprintf(mesg,"Error - Could not complete SSL handshake.");
					close(sock);
					return(1);
	            }
	        } else {
	            sprintf(mesg,"Could not create SSL connection structure.");
				close(sock);
				return(1);
	        }
		}
#endif
		/* make the query */
		if(debugmode) { printf("Building query\n"); fflush(NULL); }
		bzero(&send_packet,sizeof(send_packet));
		randomize_buffer((char *)&send_packet,sizeof(send_packet));
		send_packet.packet_version=(int16_t)htons(NRPE_PACKET_VERSION_2);
        send_packet.packet_type=(int16_t)htons(QUERY_PACKET);

		if(debugmode) { printf("-> [%s]",cmd); fflush(NULL); }
        strncpy(&send_packet.buffer[0],cmd,MAX_PACKETBUFFER_LENGTH);
		for( i=0; i<MAXARGS; i++ ) {
			if(arg[i]) {
				if(debugmode) { printf(" [%s]",arg[i]); fflush(NULL); }
				strncat(&send_packet.buffer[0],"!",MAX_PACKETBUFFER_LENGTH);
				strncat(&send_packet.buffer[0],arg[i],MAX_PACKETBUFFER_LENGTH);
			}
		}
		if(debugmode) { printf("\n"); fflush(NULL); }

        send_packet.buffer[MAX_PACKETBUFFER_LENGTH-1]='\x0';
		send_packet.crc32_value=(u_int32_t)0L;
        calculated_crc32=calculate_crc32((char *)&send_packet,sizeof(send_packet));
        send_packet.crc32_value=(u_int32_t)htonl(calculated_crc32);

		/* send the query */
        bytes_to_send=sizeof(send_packet);
#ifdef WITHSSL
        if(usessl) {
			if(debugmode) { printf("Sending data using SSL\n"); fflush(NULL); }
            rc=SSL_write(ssl,&send_packet,bytes_to_send);
            if(rc<0) { rc=-1; } else {
				if(debugmode) { printf("Sent %d bytes from %d\n",
					rc,bytes_to_send); fflush(NULL); }
			}
		} else {
			if(debugmode) { printf("Sending data\n"); fflush(NULL); }
        	rc=sendall(sock,(char *)&send_packet,&bytes_to_send);
		}
#else
		if(debugmode) { printf("Sending data\n"); fflush(NULL); }
        rc=sendall(sock,(char *)&send_packet,&bytes_to_send);
#endif
		if(rc<0) {
			sprintf(mesg,"Error on command send.");
			close(sock);
			return(1);
		}
		/* get the response */
        bytes_to_recv=sizeof(receive_packet);
		if(debugmode) { printf("Receiving data (%d bytes)\n",
			bytes_to_recv); fflush(NULL); }
#ifdef WITHSSL
		if(usessl) {
            rc=SSL_read(ssl,&receive_packet,bytes_to_recv);
			if(debugmode) { 
				printf("Closing down SSL connection\n"); fflush(NULL); }
		    SSL_shutdown(ssl);
		    SSL_free(ssl);
		} else {
            rc=recvall(sock,(char *)&receive_packet,&bytes_to_recv,timeout);
		}
#else
        rc=recvall(sock,(char *)&receive_packet,&bytes_to_recv,timeout);
#endif
		if(debugmode) { printf("Closing down socket\n"); fflush(NULL); }
		close(sock); sock=-1;
		if(rc<0) {
			sprintf(mesg,"Error on response read.");
			return(1);
		}
		if(!rc) {
			sprintf(mesg,"Received 0 bytes: Check NRPE only_from and tcp_wrappers rules.");
			return(1);
		}
		if(bytes_to_recv<sizeof(receive_packet)) {
			sprintf(mesg,"Incorrect packet size received.");
			return(1);
		}

		/* test data */
		if(debugmode) { printf("Testing data\n"); fflush(NULL); }
		packet_crc32=ntohl(receive_packet.crc32_value);
        receive_packet.crc32_value=0L;
        calculated_crc32=calculate_crc32((char *)&receive_packet,sizeof(receive_packet));
        if(packet_crc32!=calculated_crc32){
			sprintf(mesg,"Incorrect checksum received. Is this really NRPE?");
			return(1);
		}
		if(ntohs(receive_packet.packet_type)!=RESPONSE_PACKET){
			sprintf(mesg,"Incorrect packet type.  Is this really NRPE?");
			return(1);
		}

		result=(int16_t)ntohs(receive_packet.result_code);
        receive_packet.buffer[MAX_PACKETBUFFER_LENGTH-1]='\x0';
		snprintf(mesg,sizeof(mesg),"(%d) %s",result,receive_packet.buffer);
		if((result<0) || (result>2)) { return(1); }
		buf = receive_packet.buffer;

		if( cachemode ) {
			newdata = 1; 
			cacheadd(cmd, buf);
		}

		} /* used cached data */

		strncpy(mesg,buf,sizeof(mesg));

		/* parse the buffer */
		if(debugmode) { printf("Parsing data\n[%s]\n",buf); fflush(NULL); }
		s1 = s2 = buf;
		if(perfmode) {
			s1 = strchr(buf,'|');
			if(s1) { s2 = s1; } else { s1 = s2; }
		}
		while( *s1 && (parsed < MAXPARSE) ) {
			while( *s1 && ((*s1 < '0')||(*s1 > '9'))) { s1++; }
			s2 = s1;
			while(*s2 && (((*s2>='0') && (*s2<='9'))||(*s2=='.')) ){ s2++; }
			if(*s2) { *s2 = '\0'; s2++; }
			a[parsed++] = atof(s1);
			s1 = s2;
		}
		if(debugmode) {
			printf("Processed: ");
			for(i=0; (i<MAXPARSE)&&(i<parsed); i++) { printf("%.2f,",a[i]); }
			printf("\n");
		}
		if(o>parsed) { return(1); }
		*rvp = a[o];
	}
	/* report it all */
	if(debugmode) { printf ("Returning %f\n",*rvp);fflush(NULL);	}
	return(0);
}
#ifdef WITHALARM
void handler(int c) {
	printf("UNKNOWN\nUNKNWON\n\nTimeout (%dsec) on query.\n",timeout);
	exit(1);
}
#endif
/*
######################################################################
# defaults
*/
int main(int argc, char **argv) {
int c;
int n,nn;
int hasoffset = 0;

port = PORT;
host[0] = '\0';
resp[0] = resp[1] = 0;
unknown[0] = unknown[1] = 1;
mesg[0] = '\0';

/* process arguments */
static struct option options[] = {
	{ "host", 1, 0, 'H' },
	{ "port", 1, 0, 'p' },
	{ "offset", 1, 0, 'o' },
	{ "module", 1, 0, 'c' },
	{ "command", 1, 0, 'c' },
	{ "cmd", 1, 0, 'c' },
	{ "arg", 1, 0, 'a' },
	{ "debug", 0, 0, 'd' },
	{ "timeout", 1, 0, 't' },
	{ "nossl", 0, 0, 'n' },
	{ "perfparse", 0, 0, 'x' },
	{ "cache", 0, 0, 'C' }
};
while(1) {
	c = getopt_long(argc,argv,"CMnxP:H:s:p:o:n:c:v:l:a:b:dt:h",options,NULL);
	if(c == -1) break;
	if(debugmode) { printf("Recording option=[%c]\n",c); fflush(NULL); }
	switch(c) {
		case 'M': break; /* backwards compatibility */
		case 'C': /* cache mode on */
			cachemode = 1; break;
		case 'H':
		case 's':
			strncpy(host,optarg,sizeof(host)-1); break;
		case 'p':
			port = atoi(optarg); break;
		case 'o': /* offset */
			n = 0; 
			if(hasoffset || cmd[1] || args[1][0]) { n = 1; } 
/*			if(hasoffset ) { if(debugmode){printf("hasoffset\n"); } n = 1; }
			if(cmd[1]    ) { if(debugmode){printf("cmd[1]!=0\n"); } n = 1; }
			if(args[1][0]) { if(debugmode){printf("argc[1][0]\n"); } n = 1; }*/
			offset[n] = atoi(optarg);
			if(debugmode) { printf("Saving offset=[%s]=%d to slot %d\n",optarg,offset[n],n); }
			hasoffset = 1;
			break;
		case 'n':
			usessl = 0; break;
		case 'c': /* command */
		case 'v':
			n = 0; if(cmd[n]) { n = 1; offset[1] = 0; }
			if(cmd[n]) {
				printf("You may only specify two commands.\n");
				exit(1);
			}
			cmd[n] = optarg;
			break;
		case 'l': /* arg */
		case 'a':
		case 'b':
			n = 0; if(cmd[1] || (c=='b')) { n = 1; }
			if(n && !cmd[n]) { cmd[n] = cmd[0]; }
			for( nn = 0; args[n][nn] && (nn<MAXARGS); nn++ );
			if(nn<MAXARGS) { args[n][nn]=optarg; }
			else { printf("Too many arguments for %s\n",cmd[n]); exit(1); }
			break;
		case 'd':
			debugmode = 1; break;
		case 't':
			timeout = atoi(optarg);
			if(timeout < 1) { timeout = TIMEOUT; }	
			break;
		case 'x':
			perfmode = 1; break;
		case 'h':
			dohelp(); exit(1);
		default:
			printf("Option was not recognised...\n"); exit(1);
	} /* switch */
} /* while loop */

if( !port || !host[0] ) {
	sprintf(mesg,"Must specify a valid port and hostname");
	outputresp();
	exit( 1 );
}

/* we need to run a second command only if the args have changed */
if(!cmd[1] && args[1][0]) { cmd[1] = cmd[0]; }

if(!cmd[0]) {
	sprintf(mesg,"No command was given.");
	outputresp(); exit(1);
}

generate_crc32_table();
#ifdef WITHSSL
/* initialize SSL */
if(usessl==TRUE){
    SSL_library_init();
    SSLeay_add_ssl_algorithms();
    meth=SSLv23_client_method();
    SSL_load_error_strings();
    if((ctx=SSL_CTX_new(meth))==NULL){
        sprintf(mesg,"Error - could not create SSL context.");
		outputresp(); exit(0);
    }
    SSL_CTX_set_options(ctx,SSL_OP_NO_SSLv2 | SSL_OP_NO_SSLv3);
}
#endif

if(cachemode) {
	strcpy(cachefile,CACHEFILE);
}

/*
# Now we have one or two command to pass to the agent.
# We connect, and send, then listen for the response.
# Repeat for second argument if necessary
*/
#ifdef WITHALARM
/* timeout for program */
signal(SIGALRM,handler);
if(debugmode) { printf("Starting alarm for %d sec\n",timeout); fflush(NULL); }
alarm(timeout);
#endif
/* Connect */
if(debugmode) { printf("Starting queries\n"); fflush(NULL); }
if(cachemode) { readcache(); }

n = ask(cmd[0],args[0],&resp[0],offset[0],perfmode,0);
if(n) { outputresp(); exit(0); }
else { unknown[0] = 0; }
if(cmd[1]) {
	n = ask(cmd[1],args[1],&resp[1],offset[1],perfmode,0);
	if(n) { outputresp(); exit(0); }
	else { unknown[1] = 0; }
} else {
	n = ask((char *)0,(char **)0,&resp[1],offset[1],perfmode,1);
	if(!n) { unknown[1] = 0; }
}
#ifdef WITHALARM
alarm(0);
#endif

if(!mesg[0]) {
	sprintf(mesg,"MRTG-NRPE query agent version %s",VERSION);
}

#ifdef WITHSSL
if(usessl) { SSL_CTX_free(ctx); }
#endif

outputresp();
exit(0);
}
